/******************************************************************************		
*                                                                                   	
*       Copyright (C) 2009 Freescale Semiconductor, Inc.                            	
*       All Rights Reserved								              				
*														              					
* Filename:       adc_flowtron.c               											
*														              					
* Description:	  16-bit ADC Driver file for MC9S08GW64	        	    	  	       						
*	
* Author:         Tanya Malik																					
* Notes:        																		
******************************************************************************/		

/******************************************************************************/
/* Include Files                                                              */
/*****************************************************************************/
#include "MC9S08GW64.h"
#include "pdb_flowtron.h" 

 void (*pdb_callback)(unsigned char);  //This is a pointer to the function which 
                                                     //the user needs to pass in SCI_Init
                                                     //so that the required function is 
                                                     //called in interrupt subroutine



/******************************************************************************/
/* FUNCTION          : PDB_Init                                               */
/******************************************************************************/
/* Abstract          : This function initializes pdb interface                */
/*					           by configuring the internal registers                  */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.Prescaler - PDB_PRESCALER_1/PDB_PRESCALER_2          */
/*                                  /PDB_PRESCALER_4/PDB_PRESCALER_8          */
/*                                  /PDB_PRESCALER_16/PDB_PRESCALER_32        */
/*                                  /PDB_PRESCALER_64/PDB_PRESCALER_128       */
/*                     2.Continuous_OneShot_Mode - PDB_CONT_MODE              */
/*                                                 /PDB_ONE_SHOT_MODE         */
/*                     3.Trig_Select - PDB_COMP1_TRIG_SEL/PDB_COMP2_TRIG_SEL  */
/*                                     /PDB_EX_TRIG_SEL/PDB_SW_TRIG_SEL       */
/*                     4.ADC_Mode - MODE_8/MODE_10/MODE_12/MODE_16            */
/*                     5.Idelay_Interrupt_Enable - PDB_IDELAY_INT_ENABLE      */
/*                                                 /PDB_IDELAY_INT_DISABLE    */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Init(unsigned char Prescaler,unsigned char Continuous_OneShot_Mode,
              unsigned char Trig_Select,unsigned char Idelay_Interrupt_Enable,
              void (*p)(unsigned char)) 
{
  pdb_callback=p;
  
  SCGC4_PDB = 1;             //enables the clock to the PDB module
 
  PTAPF3 = 0x30;             //enables the pin muxing for EXTRIG  
 
  PDBSCH_PRESCALER = Prescaler;  //sets the value of prescaler for PDB clock
  PDBSCL_CONT= Continuous_OneShot_Mode;  //sets if the mode is one shot or continuous
  PDBSCL_TRIGSEL =  Trig_Select;          //selects between Software trigger, external trigger,comparator 1 , comparator 2
  PDBSCL_IE = Idelay_Interrupt_Enable;    //selects if the interrupt is enabled for idelay match
  PDBSCL_EN = 1;  //enables the PDB module
  
}

/******************************************************************************/
/* FUNCTION          : PDB_Set_Mod_Idelay                                     */
/******************************************************************************/
/* Abstract          : This function sets the Mod value and Idelay of the PDB */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.Mod_Value  - enter the Mod value of the counter      */
/*                     2.Idelay  - set the idelay value after which the       */
/*                                 interrupt is required                      */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Set_Mod_Idelay(unsigned int Mod_Value,unsigned int Idelay) 
{
  PDBMOD = Mod_Value;           //Enable Mod reg
  PDBIDLY = Idelay; 
}

/******************************************************************************/
/* FUNCTION          : PDB_Set_Ch1_Delay_A                                    */
/******************************************************************************/
/* Abstract          : This function sets the delay of channel1 after which   */
/*                     Pretrigger A is required                               */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ChannelA_Delay  - enter the delay for pretrigger A   */
/*                                         in channel 1                       */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Set_Ch1_Delay_A(unsigned int ChannelA_Delay) 
{
  PDBCH1CRL_AOS = 1;
  PDBCH1CRL_ENA = 1;
  PDBCH1DLYA = ChannelA_Delay;
}

/******************************************************************************/
/* FUNCTION          : PDB_Set_Ch1_Delay_B                                    */
/******************************************************************************/
/* Abstract          : This function sets the delay of channel1 after which   */
/*                     Pretrigger B is required                               */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ChannelB_Delay  - enter the delay for pretrigger B   */
/*                                         in channel 1                       */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Set_Ch1_Delay_B(unsigned int ChannelB_Delay) 
{
  PDBCH1CRL_BOS = 1;
  PDBCH1CRL_ENB = 1;
  PDBCH1DLYB = ChannelB_Delay;
}

/******************************************************************************/
/* FUNCTION          : PDB_Set_Ch2_Delay_A                                    */
/******************************************************************************/
/* Abstract          : This function sets the delay of channel2 after which   */
/*                     Pretrigger A is required                               */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ChannelA_Delay  - enter the delay for pretrigger A   */
/*                                         in channel 2                       */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Set_Ch2_Delay_A(unsigned int ChannelA_Delay) 
{
  PDBCH2CRL_AOS = 1;
  PDBCH2CRL_ENA = 1;
  PDBCH2DLYA = ChannelA_Delay;
}

/******************************************************************************/
/* FUNCTION          : PDB_Set_Ch1_Delay_B                                    */
/******************************************************************************/
/* Abstract          : This function sets the delay of channel2 after which   */
/*                     Pretrigger B is required                               */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ChannelB_Delay  - enter the delay for pretrigger B   */
/*                                         in channel 2                       */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Set_Ch2_Delay_B(unsigned int ChannelB_Delay) 
{
  PDBCH2CRL_BOS = 1;
  PDBCH2CRL_ENB = 1;
  PDBCH2DLYB = ChannelB_Delay;
}

/******************************************************************************/
/* FUNCTION          : PDB_Start_SW_Trigger                                   */
/******************************************************************************/
/* Abstract          : This function starts the conversion by givin software  */
/*                     trigger (if trigger select is PDB_SW_TRIG_SEL          */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : None
/* Return Parameter  : None                                                   */
/******************************************************************************/

void PDB_Start_SW_Trigger()
{
	PDBSCL_SWTRIG = 1;
}

/*interrupt subroutines*/

void interrupt VectorNumber_Vpdb pdb_isr(void) 
{
  PDBSCL_IF = 1; //clears the interrupt
  if(pdb_callback != 0)
        (*pdb_callback)(PDB_IDELAY_INTERRUPT);
      //sends the data to a function pointed by sci_callback
}

void interrupt VectorNumber_Vpdb_err pdb_isr_err(void) 
{
  if(PDBCH1CR & 0x8000)
    {
      PDBCH1CR |= 0x8000; 
      if(pdb_callback != 0)
        (*pdb_callback)(PDB_CHANNEL_1_TRIG_A);
      //sends the data to a function pointed by sci_callback
    } 
  else 
    {
      if(PDBCH1CR & 0x4000) 
        {
          PDBCH1CR |= 0x4000;
          if(pdb_callback != 0)
           (*pdb_callback)(PDB_CHANNEL_1_TRIG_B);
      //sends the data to a function pointed by sci_callback

        }
    }
  
  if(PDBCH2CR & 0x8000)
    {
      PDBCH2CR |= 0x8000;
      if(pdb_callback != 0)
        (*pdb_callback)(PDB_CHANNEL_2_TRIG_A);
      //sends the data to a function pointed by sci_callback
     
    } 
  else
    {
      if(PDBCH2CR & 0x4000) 
      {
        PDBCH2CR |= 0x4000;      
        if(pdb_callback != 0)
        (*pdb_callback)(PDB_CHANNEL_2_TRIG_B);
      //sends the data to a function pointed by sci_callback

      }
    }
}
